<?php
/**
 * Aggressor Adventures Document Library - PHP 5 Compatible Version
 * Compatible with PHP 5.3+
 */

// Error reporting for debugging (remove in production)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Configuration - using older PHP syntax
$DOCS_DIR = dirname(__FILE__) . '/docs';
$ALLOWED_EXTENSIONS = array('pdf');
$METADATA_EXTENSIONS = array('json', 'txt');

// Simple function to scan documents - PHP 5 compatible
function scanDocuments($baseDir, $allowedExt, $metaExt) {
    $library = array();
    
    if (!is_dir($baseDir)) {
        return $library;
    }
    
    $categories = scandir($baseDir);
    if ($categories === false) {
        return $library;
    }
    
    foreach ($categories as $category) {
        if ($category === '.' || $category === '..') {
            continue;
        }
        
        $categoryPath = $baseDir . '/' . $category;
        
        if (!is_dir($categoryPath)) {
            continue;
        }
        
        $documents = array();
        $files = scandir($categoryPath);
        
        if ($files === false) {
            continue;
        }
        
        foreach ($files as $file) {
            if ($file === '.' || $file === '..') {
                continue;
            }
            
            $filePath = $categoryPath . '/' . $file;
            $fileInfo = pathinfo($file);
            
            // Check if file has extension and is allowed
            if (!isset($fileInfo['extension'])) {
                continue;
            }
            
            $extension = strtolower($fileInfo['extension']);
            if (!in_array($extension, $allowedExt)) {
                continue;
            }
            
            $description = '';
            // Try to get description from metadata files
            foreach ($metaExt as $ext) {
                $metadataFile = $categoryPath . '/' . $fileInfo['filename'] . '.' . $ext;
                if (file_exists($metadataFile)) {
                    $content = file_get_contents($metadataFile);
                    if ($content !== false) {
                        if ($ext === 'json') {
                            $data = json_decode($content, true);
                            if ($data !== null && isset($data['description'])) {
                                $description = $data['description'];
                            }
                        } else {
                            $description = trim($content);
                        }
                        break;
                    }
                }
            }
            
            $document = array(
                'filename' => $fileInfo['filename'],
                'file' => $file,
                'path' => $filePath,
                'relative_path' => $category . '/' . $file,
                'size' => filesize($filePath),
                'modified' => filemtime($filePath),
                'description' => $description
            );
            
            $documents[] = $document;
        }
        
        if (!empty($documents)) {
            $library[$category] = $documents;
        }
    }
    
    return $library;
}

// Handle PDF serving - PHP 5 compatible
if (isset($_GET['pdf']) && !empty($_GET['pdf'])) {
    $requestedPath = $_GET['pdf'];
    $pdfPath = $DOCS_DIR . '/' . $requestedPath;
    
    // Debug information (remove in production)
    if (isset($_GET['debug'])) {
        echo "<h3>Debug Information</h3>";
        echo "Requested path: " . htmlspecialchars($requestedPath) . "<br>";
        echo "Full PDF path: " . htmlspecialchars($pdfPath) . "<br>";
        echo "DOCS_DIR: " . htmlspecialchars($DOCS_DIR) . "<br>";
        echo "File exists: " . (file_exists($pdfPath) ? 'Yes' : 'No') . "<br>";
        echo "File readable: " . (is_readable($pdfPath) ? 'Yes' : 'No') . "<br>";
        $realPath = realpath($pdfPath);
        echo "Real path: " . htmlspecialchars($realPath ? $realPath : 'false') . "<br>";
        exit;
    }
    
    // Security check - PHP 5 compatible
    $realDocsDir = realpath($DOCS_DIR);
    $realPdfPath = realpath($pdfPath);
    
    if ($realPdfPath !== false && $realDocsDir !== false && strpos($realPdfPath, $realDocsDir) === 0) {
        if (file_exists($realPdfPath) && is_readable($realPdfPath)) {
            $fileInfo = pathinfo($realPdfPath);
            if (isset($fileInfo['extension']) && strtolower($fileInfo['extension']) === 'pdf') {
                // Serve the PDF
                header('Content-Type: application/pdf');
                header('Content-Length: ' . filesize($realPdfPath));
                header('Content-Disposition: inline; filename="' . basename($realPdfPath) . '"');
                
                // Use readfile for PHP 5 compatibility
                if (readfile($realPdfPath) === false) {
                    header('HTTP/1.1 500 Internal Server Error');
                    echo 'Error reading file';
                }
                exit;
            }
        }
    }
    
    // Enhanced error message for debugging
    header('HTTP/1.1 404 Not Found');
    echo "<h3>File Not Found</h3>";
    echo "Requested: " . htmlspecialchars($requestedPath) . "<br>";
    echo "Looking for: " . htmlspecialchars($pdfPath) . "<br>";
    echo "File exists: " . (file_exists($pdfPath) ? 'Yes' : 'No') . "<br>";
    echo "<a href='?debug=1&pdf=" . urlencode($requestedPath) . "'>Debug this request</a>";
    exit;
}

// Scan document library
$documentLibrary = scanDocuments($DOCS_DIR, $ALLOWED_EXTENSIONS, $METADATA_EXTENSIONS);

// Simple file size formatting - PHP 5 compatible
function formatBytes($size) {
    $units = array('B', 'KB', 'MB', 'GB');
    $unitIndex = 0;
    
    while ($size > 1024 && $unitIndex < count($units) - 1) {
        $size = $size / 1024;
        $unitIndex++;
    }
    
    return round($size, 1) . ' ' . $units[$unitIndex];
}

// HTML escape function for PHP 5
function escapeHtml($text) {
    return htmlspecialchars($text, ENT_QUOTES, 'UTF-8');
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Aggressor Adventures - Document Library</title>
    <style>
        /* Inline CSS for maximum compatibility */
        * { 
            margin: 0; 
            padding: 0; 
            -webkit-box-sizing: border-box;
            -moz-box-sizing: border-box;
            box-sizing: border-box; 
        }
        
        body {
            font-family: Arial, Helvetica, sans-serif;
            line-height: 1.6;
            color: #2d3748;
            background-color: #f7fafc;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 20px;
        }
        
        .header {
            background: #1e3a5f; /* Fallback for older browsers */
            background: -webkit-linear-gradient(135deg, #1e3a5f 0%, #2c5282 100%);
            background: -moz-linear-gradient(135deg, #1e3a5f 0%, #2c5282 100%);
            background: linear-gradient(135deg, #1e3a5f 0%, #2c5282 100%);
            color: white;
            padding: 1rem 0;
            -webkit-box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            -moz-box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        
        .header h1 {
            font-size: 1.8rem;
            margin-bottom: 0.25rem;
        }
        
        .tagline {
            font-size: 0.9rem;
            opacity: 0.9;
        }
        
        .main {
            padding: 2rem 0;
        }
        
        .search-section {
            margin-bottom: 2rem;
            text-align: center;
        }
        
        .search-input {
            width: 100%;
            max-width: 500px;
            padding: 1rem;
            border: 2px solid #e2e8f0;
            -webkit-border-radius: 8px;
            -moz-border-radius: 8px;
            border-radius: 8px;
            font-size: 1rem;
        }
        
        .search-input:focus {
            outline: none;
            border-color: #2c5282;
        }
        
        .category {
            background-color: white;
            -webkit-border-radius: 8px;
            -moz-border-radius: 8px;
            border-radius: 8px;
            -webkit-box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            -moz-box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            margin-bottom: 2rem;
            overflow: hidden;
        }
        
        .category-header {
            background: #e6f3ff; /* Fallback */
            background: -webkit-linear-gradient(135deg, #e6f3ff 0%, white 100%);
            background: -moz-linear-gradient(135deg, #e6f3ff 0%, white 100%);
            background: linear-gradient(135deg, #e6f3ff 0%, white 100%);
            padding: 1.5rem;
            border-bottom: 1px solid #e2e8f0;
            cursor: pointer;
        }
        
        .category-title {
            font-size: 1.4rem;
            color: #1e3a5f;
            margin: 0;
        }
        
        .documents-grid {
            display: block; /* Fallback for older browsers */
            padding: 1.5rem;
        }
        
        /* Modern browsers with grid support */
        @supports (display: grid) {
            .documents-grid {
                display: grid;
                grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
                gap: 1.5rem;
            }
        }
        
        .document-card {
            border: 1px solid #e2e8f0;
            -webkit-border-radius: 8px;
            -moz-border-radius: 8px;
            border-radius: 8px;
            padding: 1.5rem;
            margin-bottom: 1.5rem; /* For browsers without grid */
            cursor: pointer;
            -webkit-transition: all 0.3s ease;
            -moz-transition: all 0.3s ease;
            transition: all 0.3s ease;
        }
        
        .document-card:hover {
            border-color: #2c5282;
            -webkit-box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            -moz-box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            -webkit-transform: translateY(-2px);
            -moz-transform: translateY(-2px);
            transform: translateY(-2px);
        }
        
        .document-title {
            font-size: 1.1rem;
            font-weight: 600;
            color: #1e3a5f;
            margin-bottom: 0.5rem;
        }
        
        .document-description {
            color: #718096;
            font-size: 0.9rem;
            margin-bottom: 1rem;
        }
        
        .document-meta {
            font-size: 0.8rem;
            color: #718096;
            margin-bottom: 1rem;
        }
        
        .document-view-btn {
            width: 100%;
            background: #38a169; /* Fallback */
            background: -webkit-linear-gradient(135deg, #38a169 0%, #2f855a 100%);
            background: -moz-linear-gradient(135deg, #38a169 0%, #2f855a 100%);
            background: linear-gradient(135deg, #38a169 0%, #2f855a 100%);
            color: white;
            border: none;
            padding: 0.75rem;
            -webkit-border-radius: 8px;
            -moz-border-radius: 8px;
            border-radius: 8px;
            font-weight: 600;
            cursor: pointer;
        }
        
        .document-view-btn:hover {
            -webkit-transform: translateY(-1px);
            -moz-transform: translateY(-1px);
            transform: translateY(-1px);
        }
        
        .empty-state {
            text-align: center;
            padding: 4rem 2rem;
            background-color: white;
            -webkit-border-radius: 8px;
            -moz-border-radius: 8px;
            border-radius: 8px;
            -webkit-box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            -moz-box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        
        .footer {
            background-color: #1e3a5f;
            color: white;
            text-align: center;
            padding: 1.5rem 0;
            margin-top: 3rem;
        }
        
        /* Responsive design for older browsers */
        @media (max-width: 768px) {
            .container {
                padding: 0 15px;
            }
            
            .documents-grid {
                padding: 1rem;
            }
            
            .document-card {
                margin-bottom: 1rem;
            }
        }
    </style>
</head>
<body>
    <header class="header">
        <div class="container">
            <h1>Aggressor Adventures</h1>
            <span class="tagline">Document Library</span>
        </div>
    </header>

    <main class="main">
        <div class="container">
            <div class="search-section">
                <input type="text" id="searchInput" class="search-input" placeholder="Search documents..." onkeyup="searchDocuments()">
            </div>

            <div class="library-section">
                <?php if (empty($documentLibrary)): ?>
                    <div class="empty-state">
                        <h2>No Documents Found</h2>
                        <p>Please add PDF documents to the docs/ directory to get started.</p>
                        <p><strong>Debug Info:</strong></p>
                        <p>Docs directory: <?php echo escapeHtml($DOCS_DIR); ?></p>
                        <p>Directory exists: <?php echo is_dir($DOCS_DIR) ? 'Yes' : 'No'; ?></p>
                        <p>Directory readable: <?php echo is_readable($DOCS_DIR) ? 'Yes' : 'No'; ?></p>
                        <p>PHP Version: <?php echo phpversion(); ?></p>
                    </div>
                <?php else: ?>
                    <?php foreach ($documentLibrary as $categoryName => $documents): ?>
                        <div class="category">
                            <div class="category-header" onclick="toggleCategory(this)">
                                <h2 class="category-title">
                                    📁 <?php echo escapeHtml($categoryName); ?>
                                    (<?php echo count($documents); ?>)
                                </h2>
                            </div>

                            <div class="documents-grid">
                                <?php foreach ($documents as $document): ?>
                                    <div class="document-card" onclick="viewDocument('<?php echo escapeHtml($document['relative_path']); ?>', '<?php echo escapeHtml($document['filename']); ?>')">
                                        <div style="font-size: 2rem; text-align: center; margin-bottom: 1rem;">📄</div>
                                        <h3 class="document-title"><?php echo escapeHtml($document['filename']); ?></h3>
                                        <?php if (!empty($document['description'])): ?>
                                            <p class="document-description"><?php echo escapeHtml($document['description']); ?></p>
                                        <?php endif; ?>
                                        <div class="document-meta">
                                            <span><?php echo formatBytes($document['size']); ?></span>
                                            <span style="float: right;"><?php echo date('M j, Y', $document['modified']); ?></span>
                                            <div style="clear: both;"></div>
                                        </div>
                                        <button class="document-view-btn">View PDF</button>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
    </main>

    <footer class="footer">
        <div class="container">
            <p>&copy; <?php echo date('Y'); ?> Aggressor Adventures. All rights reserved.</p>
        </div>
    </footer>

    <script>
        // JavaScript compatible with older browsers
        function searchDocuments() {
            var input = document.getElementById('searchInput');
            var filter = input.value.toLowerCase();
            var cards = document.getElementsByClassName('document-card');

            for (var i = 0; i < cards.length; i++) {
                var title = cards[i].getElementsByClassName('document-title')[0];
                var description = cards[i].getElementsByClassName('document-description')[0];

                var titleText = title ? title.textContent || title.innerText : '';
                var descText = description ? description.textContent || description.innerText : '';

                titleText = titleText.toLowerCase();
                descText = descText.toLowerCase();

                if (titleText.indexOf(filter) > -1 || descText.indexOf(filter) > -1) {
                    cards[i].style.display = '';
                } else {
                    cards[i].style.display = 'none';
                }
            }
        }

        function toggleCategory(header) {
            var grid = header.nextSibling;
            // Find the next element node (skip text nodes)
            while (grid && grid.nodeType !== 1) {
                grid = grid.nextSibling;
            }

            if (grid) {
                if (grid.style.display === 'none') {
                    grid.style.display = 'block';
                } else {
                    grid.style.display = 'none';
                }
            }
        }

        function viewDocument(path, title) {
            // Open in new window/tab for better compatibility
            window.open('?pdf=' + encodeURIComponent(path), '_blank');
        }
    </script>
</body>
</html>
