<?php
/**
 * Aggressor Adventures Document Library
 * Self-contained LAMP application for serving PDF documents
 */

// Error reporting for debugging (remove in production)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Configuration
define('DOCS_DIR', __DIR__ . '/docs');
define('ALLOWED_EXTENSIONS', array('pdf'));
define('METADATA_EXTENSIONS', array('json', 'txt'));

// Security headers
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: SAMEORIGIN');
header('X-XSS-Protection: 1; mode=block');

/**
 * Scan directory structure and build document library
 */
function scanDocumentLibrary($baseDir) {
    $library = [];
    
    if (!is_dir($baseDir)) {
        return $library;
    }
    
    $categories = scandir($baseDir);
    
    foreach ($categories as $category) {
        if ($category === '.' || $category === '..') {
            continue;
        }
        
        $categoryPath = $baseDir . '/' . $category;
        
        if (!is_dir($categoryPath)) {
            continue;
        }
        
        $documents = [];
        $files = scandir($categoryPath);
        
        foreach ($files as $file) {
            if ($file === '.' || $file === '..') {
                continue;
            }
            
            $filePath = $categoryPath . '/' . $file;
            $fileInfo = pathinfo($file);
            
            if (!isset($fileInfo['extension']) ||
                !in_array(strtolower($fileInfo['extension']), ALLOWED_EXTENSIONS)) {
                continue;
            }
            
            $document = array(
                'filename' => $fileInfo['filename'],
                'file' => $file,
                'path' => $filePath,
                'relative_path' => 'docs/' . $category . '/' . $file,
                'size' => filesize($filePath),
                'modified' => filemtime($filePath),
                'description' => getDocumentDescription($categoryPath, $fileInfo['filename'])
            );
            
            $documents[] = $document;
        }
        
        if (!empty($documents)) {
            // Sort documents by filename
            usort($documents, 'sortDocumentsByFilename');

            $library[$category] = $documents;
        }
    }
    
    return $library;
}

/**
 * Sort documents by filename (for older PHP versions)
 */
function sortDocumentsByFilename($a, $b) {
    return strcmp($a['filename'], $b['filename']);
}

/**
 * Get document description from metadata file
 */
function getDocumentDescription($categoryPath, $filename) {
    foreach (METADATA_EXTENSIONS as $ext) {
        $metadataFile = $categoryPath . '/' . $filename . '.' . $ext;
        
        if (file_exists($metadataFile)) {
            $content = file_get_contents($metadataFile);
            
            if ($ext === 'json') {
                $data = json_decode($content, true);
                return isset($data['description']) ? $data['description'] : '';
            } else {
                return trim($content);
            }
        }
    }
    
    return '';
}

/**
 * Serve PDF file with proper headers
 */
function servePDF($filePath) {
    if (!file_exists($filePath) || !is_readable($filePath)) {
        http_response_code(404);
        die('File not found');
    }
    
    // Validate file extension
    $fileInfo = pathinfo($filePath);
    if (!isset($fileInfo['extension']) || 
        strtolower($fileInfo['extension']) !== 'pdf') {
        http_response_code(403);
        die('Invalid file type');
    }
    
    // Set headers for PDF
    header('Content-Type: application/pdf');
    header('Content-Length: ' . filesize($filePath));
    header('Content-Disposition: inline; filename="' . basename($filePath) . '"');
    header('Cache-Control: public, max-age=3600');
    
    readfile($filePath);
    exit;
}

// Handle PDF serving
if (isset($_GET['pdf']) && !empty($_GET['pdf'])) {
    $pdfPath = DOCS_DIR . '/' . $_GET['pdf'];
    
    // Security check: ensure path is within docs directory
    $realDocsDir = realpath(DOCS_DIR);
    $realPdfPath = realpath($pdfPath);
    
    if ($realPdfPath && strpos($realPdfPath, $realDocsDir) === 0) {
        servePDF($realPdfPath);
    } else {
        http_response_code(403);
        die('Access denied');
    }
}

// Scan document library
$documentLibrary = scanDocumentLibrary(DOCS_DIR);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Aggressor Adventures - Document Library</title>
    <link rel="stylesheet" href="assets/style.css">
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Open+Sans:wght@300;400;600;700&display=swap" rel="stylesheet">
</head>
<body>
    <header class="header">
        <div class="container">
            <div class="header-content">
                <div class="logo">
                    <h1>Aggressor Adventures</h1>
                    <span class="tagline">Document Library</span>
                </div>
                <nav class="nav">
                    <a href="#" class="nav-link">Safety</a>
                    <a href="#" class="nav-link">Resources</a>
                    <a href="#" class="nav-link">Contact</a>
                </nav>
            </div>
        </div>
    </header>

    <main class="main">
        <div class="container">
            <div class="search-section">
                <div class="search-container">
                    <input type="text" id="searchInput" class="search-input" placeholder="Search documents...">
                    <button class="search-button">
                        <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <circle cx="11" cy="11" r="8"></circle>
                            <path d="m21 21-4.35-4.35"></path>
                        </svg>
                    </button>
                </div>
            </div>

            <div class="library-section">
                <?php if (empty($documentLibrary)): ?>
                    <div class="empty-state">
                        <h2>No Documents Found</h2>
                        <p>Please add PDF documents to the docs/ directory to get started.</p>
                    </div>
                <?php else: ?>
                    <div class="categories">
                        <?php foreach ($documentLibrary as $categoryName => $documents): ?>
                            <div class="category" data-category="<?php echo htmlspecialchars($categoryName); ?>">
                                <div class="category-header">
                                    <h2 class="category-title">
                                        <span class="category-icon">📁</span>
                                        <?php echo htmlspecialchars($categoryName); ?>
                                        <span class="document-count">(<?php echo count($documents); ?>)</span>
                                    </h2>
                                    <button class="category-toggle" aria-expanded="true">
                                        <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                            <polyline points="6,9 12,15 18,9"></polyline>
                                        </svg>
                                    </button>
                                </div>
                                
                                <div class="documents-grid">
                                    <?php foreach ($documents as $document): ?>
                                        <div class="document-card" 
                                             data-filename="<?php echo htmlspecialchars($document['filename']); ?>"
                                             data-description="<?php echo htmlspecialchars($document['description']); ?>">
                                            <div class="document-icon">📄</div>
                                            <div class="document-info">
                                                <h3 class="document-title"><?php echo htmlspecialchars($document['filename']); ?></h3>
                                                <?php if (!empty($document['description'])): ?>
                                                    <p class="document-description"><?php echo htmlspecialchars($document['description']); ?></p>
                                                <?php endif; ?>
                                                <div class="document-meta">
                                                    <span class="document-size"><?php echo formatFileSize($document['size']); ?></span>
                                                    <span class="document-date"><?php echo date('M j, Y', $document['modified']); ?></span>
                                                </div>
                                            </div>
                                            <button class="document-view-btn" 
                                                    onclick="viewDocument('<?php echo htmlspecialchars($document['relative_path']); ?>', '<?php echo htmlspecialchars($document['filename']); ?>')">
                                                View PDF
                                            </button>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </main>

    <!-- PDF Viewer Modal -->
    <div id="pdfModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 id="pdfTitle" class="modal-title"></h3>
                <button class="modal-close" onclick="closePDFViewer()">&times;</button>
            </div>
            <div class="modal-body">
                <iframe id="pdfViewer" class="pdf-viewer" src="" frameborder="0"></iframe>
            </div>
        </div>
    </div>

    <footer class="footer">
        <div class="container">
            <p>&copy; <?php echo date('Y'); ?> Aggressor Adventures. All rights reserved.</p>
        </div>
    </footer>

    <script src="assets/script.js"></script>
</body>
</html>

<?php
/**
 * Format file size in human readable format
 */
function formatFileSize($bytes) {
    $units = ['B', 'KB', 'MB', 'GB'];
    $bytes = max($bytes, 0);
    $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
    $pow = min($pow, count($units) - 1);
    
    $bytes /= pow(1024, $pow);
    
    return round($bytes, 1) . ' ' . $units[$pow];
}
?>
