#!/bin/bash

# Aggressor Adventures Document Library Deployment Script
# This script automates the deployment process for Ubuntu/Apache servers

set -e  # Exit on any error

# Configuration
INSTALL_DIR="/var/www/html/docs"
APACHE_USER="www-data"
APACHE_GROUP="www-data"

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Function to print colored output
print_status() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

print_success() {
    echo -e "${GREEN}[SUCCESS]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

# Check if running as root
check_root() {
    if [[ $EUID -ne 0 ]]; then
        print_error "This script must be run as root (use sudo)"
        exit 1
    fi
}

# Check if Apache is installed
check_apache() {
    if ! command -v apache2 &> /dev/null; then
        print_error "Apache2 is not installed. Please install it first:"
        echo "sudo apt update && sudo apt install apache2"
        exit 1
    fi
    print_success "Apache2 is installed"
}

# Check if PHP is installed
check_php() {
    if ! command -v php &> /dev/null; then
        print_error "PHP is not installed. Please install it first:"
        echo "sudo apt update && sudo apt install php libapache2-mod-php"
        exit 1
    fi
    print_success "PHP is installed"
}

# Enable required Apache modules
enable_apache_modules() {
    print_status "Enabling required Apache modules..."
    
    modules=("rewrite" "headers" "expires" "deflate")
    
    for module in "${modules[@]}"; do
        if a2enmod "$module" &> /dev/null; then
            print_success "Enabled module: $module"
        else
            print_warning "Module $module may already be enabled"
        fi
    done
}

# Create installation directory
create_install_dir() {
    print_status "Creating installation directory: $INSTALL_DIR"
    
    if [[ -d "$INSTALL_DIR" ]]; then
        print_warning "Directory $INSTALL_DIR already exists"
        read -p "Do you want to continue? This will overwrite existing files. (y/N): " -n 1 -r
        echo
        if [[ ! $REPLY =~ ^[Yy]$ ]]; then
            print_error "Installation cancelled"
            exit 1
        fi
    else
        mkdir -p "$INSTALL_DIR"
        print_success "Created directory: $INSTALL_DIR"
    fi
}

# Copy application files
copy_files() {
    print_status "Copying application files..."
    
    # Copy main files
    cp index.php "$INSTALL_DIR/"
    cp .htaccess "$INSTALL_DIR/"
    cp README.md "$INSTALL_DIR/"
    
    # Copy assets directory
    cp -r assets "$INSTALL_DIR/"
    
    # Create docs directory if it doesn't exist
    if [[ ! -d "$INSTALL_DIR/docs" ]]; then
        mkdir -p "$INSTALL_DIR/docs"
        print_success "Created docs directory"
    fi
    
    # Copy sample docs if they exist and docs directory is empty
    if [[ -d "docs" ]] && [[ -z "$(ls -A "$INSTALL_DIR/docs" 2>/dev/null)" ]]; then
        cp -r docs/* "$INSTALL_DIR/docs/"
        print_success "Copied sample documents"
    fi
    
    print_success "Files copied successfully"
}

# Set proper permissions
set_permissions() {
    print_status "Setting file permissions..."
    
    # Set ownership
    chown -R "$APACHE_USER:$APACHE_GROUP" "$INSTALL_DIR"
    
    # Set directory permissions
    find "$INSTALL_DIR" -type d -exec chmod 755 {} \;
    
    # Set file permissions
    find "$INSTALL_DIR" -type f -exec chmod 644 {} \;
    
    # Make sure .htaccess is readable
    chmod 644 "$INSTALL_DIR/.htaccess"
    
    print_success "Permissions set successfully"
}

# Test Apache configuration
test_apache_config() {
    print_status "Testing Apache configuration..."
    
    if apache2ctl configtest &> /dev/null; then
        print_success "Apache configuration is valid"
    else
        print_error "Apache configuration test failed"
        apache2ctl configtest
        exit 1
    fi
}

# Restart Apache
restart_apache() {
    print_status "Restarting Apache..."
    
    if systemctl restart apache2; then
        print_success "Apache restarted successfully"
    else
        print_error "Failed to restart Apache"
        exit 1
    fi
}

# Display final instructions
show_final_instructions() {
    echo
    print_success "Installation completed successfully!"
    echo
    echo "Next steps:"
    echo "1. Add your PDF documents to: $INSTALL_DIR/docs/"
    echo "2. Create category folders (e.g., Safety, Dive Information, Itineraries)"
    echo "3. Place PDF files in the appropriate category folders"
    echo "4. Optionally add .json or .txt files for document descriptions"
    echo
    echo "Example structure:"
    echo "$INSTALL_DIR/docs/"
    echo "├── Safety/"
    echo "│   ├── emergency-procedures.pdf"
    echo "│   └── emergency-procedures.json"
    echo "├── Dive Information/"
    echo "│   └── dive-sites.pdf"
    echo "└── Itineraries/"
    echo "    └── 7-day-bahamas.pdf"
    echo
    echo "Access your document library at: http://your-server-ip/"
    echo
    print_warning "Remember to configure your Apache virtual host if needed!"
}

# Main installation process
main() {
    echo "Aggressor Adventures Document Library Deployment"
    echo "================================================"
    echo
    
    check_root
    check_apache
    check_php
    enable_apache_modules
    create_install_dir
    copy_files
    set_permissions
    test_apache_config
    restart_apache
    show_final_instructions
}

# Run main function
main "$@"
